<?php
// api.php
header('Content-Type: application/json; charset=utf-8');

// 配置文件路径
$configFile = __DIR__ . '/config/config.php';
if (!file_exists($configFile)) {
    http_response_code(500);
    echo json_encode(['error' => '系统未初始化'], JSON_UNESCAPED_UNICODE);
    exit;
}
$config = require $configFile;

// 连接数据库
try {
    $pdo = new PDO(
        "mysql:host=" . $config['db_host'] . ";dbname=" . $config['db_name'] . ";charset=utf8mb4",
        $config['db_user'],
        $config['db_pass']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => '数据库连接失败'], JSON_UNESCAPED_UNICODE);
    exit;
}

// 获取参数 - 现在只需要 ServerName
$serverName = isset($_GET['ServerName']) ? trim($_GET['ServerName']) : '';

if ($serverName === '') {
    echo json_encode(['error' => '缺少 ServerName 参数'], JSON_UNESCAPED_UNICODE);
    exit;
}

// 计算 base URL（用于构建 vapi 链接）
$protocol = ( (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
            || (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https')
            || (!empty($_SERVER['HTTP_FRONT_END_HTTPS']) && $_SERVER['HTTP_FRONT_END_HTTPS'] !== 'off')
            ) ? 'https' : 'http';

$host = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : ($_SERVER['SERVER_NAME'] ?? 'localhost');
$scriptDir = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\'); // 例如 "/folder" 或 ""
$baseUrl = $protocol . '://' . $host . ($scriptDir === '' ? '' : $scriptDir); // 不含尾斜杠

// 读取 duocang_data 配置
$stmt = $pdo->prepare("SELECT repos, attach_local FROM duocang_data WHERE ServerName = :ServerName LIMIT 1");
$stmt->execute([':ServerName' => $serverName]);
$row = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$row) {
    echo json_encode([
        'error'  => '对应的分组配置不存在',
        'server' => $serverName
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$result = [];
$existingNames = []; // 用于去重（避免本地与远程同名重复）

// 1) 按保存顺序处理 duocang_data.repos（存储的是仓库名称，以 ; 分隔）
if (!empty($row['repos'])) {
    // 支持分号分隔并去掉空项
    $repoNames = array_values(array_filter(array_map('trim', explode(';', $row['repos'])), function($v){ return $v !== ''; }));
    foreach ($repoNames as $name) {
        // 忽略空 name
        if ($name === '') continue;
        // 记录并输出指向 vapi 的统一入口：vapi.php?name=NAME
        $existingNames[$name] = true;
        $result[] = [
            'name' => $name,
            'url'  => $baseUrl . '/vapi.php?ServerName=' . urlencode($serverName) . '&name=' . urlencode($name)
        ];
    }
}

// 2) 如果需要附加本地源，遍历 local_files 表并查找 local_repo 中的匹配文件
if (!empty($row['attach_local'])) {
    // 读取需要查找的文件名列表
    $stmt2 = $pdo->query("SELECT filename FROM local_files");
    $localFiles = $stmt2->fetchAll(PDO::FETCH_COLUMN);

    if (!empty($localFiles)) {
        $baseDir = __DIR__ . '/local_repo';
        if (is_dir($baseDir)) {
            // 为效率，将 localFiles 转为快速查找键数组
            $targets = array_flip($localFiles);

            // 递归遍历 local_repo
            $it = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($baseDir, FilesystemIterator::SKIP_DOTS)
            );

            foreach ($it as $file) {
                if (!$file->isFile()) continue;
                $filename = $file->getFilename();
                if (!isset($targets[$filename])) continue;

                // 父目录名作为 name（例如 local_repo/tvboxqq/OK/api.json -> name = OK）
                $parentName = basename($file->getPathInfo()->getFilename() === '' ? dirname($file->getPathname()) : $file->getPathInfo()->getFilename());
                // 上述获取 parentName 更稳妥的方法：
                $parentName = basename(dirname($file->getPathname()));
                if ($parentName === '') continue;

                // 如果该 name 已存在于分组（远程仓库或已添加本地），则跳过以避免重复
                if (isset($existingNames[$parentName])) {
                    continue;
                }

                // 否则加入结果，url 也指向 vapi.php?name=NAME（vapi 内再判断该 name 是本地还是 repo）
                $existingNames[$parentName] = true;
                $result[] = [
                    'name' => $parentName,
                    'url'  => $baseUrl . '/vapi.php?ServerName=' . urlencode($serverName) . '&name=' . urlencode($parentName)
                ];
            }
        }
    }
}

// 直接输出JSON数据，不再进行加密混淆
echo json_encode(['urls' => $result], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
exit;