<?php
session_start();
// 配置文件路径
$configFile = __DIR__ . '/config/config.php';

// 如果配置文件不存在，跳转到 init.php 初始化
if (!file_exists($configFile)) {
    header('Location: init.php');
    exit;
}

// 如果未登录则跳转到 login.php
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    header('Location: login.php');
    exit;
}

// 处理登出
if (isset($_GET['action']) && $_GET['action'] === 'logout') {
    session_destroy();
    header('Location: login.php');
    exit;
}

// 加载数据库配置
$config = require __DIR__ . '/config/config.php';

try {
    $pdo = new PDO(
        "mysql:host=".$config['db_host'].";dbname=".$config['db_name'].";charset=utf8mb4",
        $config['db_user'],
        $config['db_pass']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("数据库连接失败: " . $e->getMessage());
}

// 获取统计数据
// 1. 多仓分组统计
$stmt = $pdo->query("SELECT COUNT(*) as total, 
                     SUM(attach_local) as with_local, 
                     SUM(attach_live) as with_live 
                     FROM duocang_data");
$groupStats = $stmt->fetch(PDO::FETCH_ASSOC);

// 2. 仓库统计
$stmt = $pdo->query("SELECT COUNT(*) as total FROM repos");
$repoStats = $stmt->fetch(PDO::FETCH_ASSOC);

// 3. 直播源统计
$stmt = $pdo->query("SELECT COUNT(*) as total FROM lives");
$liveStats = $stmt->fetch(PDO::FETCH_ASSOC);

// 4. 本地文件统计
$stmt = $pdo->query("SELECT COUNT(*) as total FROM local_files");
$localFileStats = $stmt->fetch(PDO::FETCH_ASSOC);

// 5. Emby配置统计
$stmt = $pdo->query("SELECT COUNT(*) as total FROM emby");
$embyStats = $stmt->fetch(PDO::FETCH_ASSOC);

// 6. 系统信息
$systemInfo = [
    'php_version' => PHP_VERSION,
    'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? '未知',
    'mysql_version' => $pdo->getAttribute(PDO::ATTR_SERVER_VERSION),
    'upload_max_filesize' => ini_get('upload_max_filesize'),
    'memory_limit' => ini_get('memory_limit'),
    'max_execution_time' => ini_get('max_execution_time'),
    'timezone' => date_default_timezone_get()
];
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>系统概览 - 多仓管理系统</title>
    <link rel="stylesheet" href="style.css">
    <style>
        /* 页面特定样式 */
        .card {
            background: var(--surface-color);
            border-radius: var(--border-radius);
            box-shadow: var(--shadow);
            padding: 25px;
            margin-bottom: 25px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
            gap: 20px;
            margin-bottom: 25px;
        }
        
        .stat-card {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            border-radius: var(--border-radius);
            padding: 20px;
            text-align: center;
            transition: var(--transition);
            cursor: pointer;
        }
        
        .stat-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }
        
        .stat-card.info {
            background: linear-gradient(135deg, #17a2b8, #138496);
        }
        
        .stat-card.purple {
            background: linear-gradient(135deg, #6f42c1, #5a2d9c);
        }
        
        .stat-card.orange {
            background: linear-gradient(135deg, #fd7e14, #e55a00);
        }
        
        .stat-card.teal {
            background: linear-gradient(135deg, #20c997, #199d76);
        }
        
        .stat-number {
            font-size: 2.2rem;
            font-weight: 700;
            margin-bottom: 5px;
            line-height: 1;
        }
        
        .stat-label {
            font-size: 0.85rem;
            opacity: 0.9;
            margin-bottom: 8px;
        }
        
        .stat-subtext {
            font-size: 0.75rem;
            opacity: 0.8;
            margin-top: 5px;
        }
        
        .dashboard-section {
            margin-bottom: 30px;
        }
        
        .section-title {
            font-size: 1.4rem;
            font-weight: 600;
            color: var(--secondary-color);
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 2px solid var(--border-color);
        }
        
        .quick-actions {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 25px;
        }
        
        .action-btn {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 15px 20px;
            background: var(--surface-color);
            border: 2px solid var(--border-color);
            border-radius: var(--border-radius);
            text-decoration: none;
            color: var(--text-primary);
            transition: var(--transition);
            font-weight: 500;
        }
        
        .action-btn:hover {
            border-color: var(--primary-color);
            color: var(--primary-color);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        .action-icon {
            font-size: 1.5rem;
            width: 40px;
            text-align: center;
        }
        
        /* 系统信息布局优化 */
        .system-info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .info-group {
            background: var(--background-color);
            border-radius: var(--border-radius);
            padding: 20px;
            border-left: 4px solid var(--primary-color);
        }
        
        .info-group-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--secondary-color);
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .info-group-title i {
            font-size: 1.2rem;
        }
        
        .info-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px 0;
            border-bottom: 1px solid var(--border-color);
        }
        
        .info-item:last-child {
            border-bottom: none;
        }
        
        .info-label {
            font-weight: 600;
            color: var(--text-secondary);
            font-size: 0.9rem;
        }
        
        .info-value {
            color: var(--text-primary);
            font-family: 'Courier New', monospace;
            font-size: 0.85rem;
            background: var(--surface-color);
            padding: 4px 8px;
            border-radius: 4px;
            border: 1px solid var(--border-color);
        }
        
        .time-display {
            text-align: center;
            padding: 15px;
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            border-radius: var(--border-radius);
            margin-top: 20px;
        }
        
        .time-label {
            font-size: 0.9rem;
            opacity: 0.9;
            margin-bottom: 5px;
        }
        
        .time-value {
            font-size: 1.3rem;
            font-weight: 600;
            font-family: 'Courier New', monospace;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 40px;
            color: var(--text-secondary);
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.5;
        }
        
        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            }
            
            .stat-number {
                font-size: 1.8rem;
            }
            
            .quick-actions {
                grid-template-columns: 1fr;
            }
            
            .system-info-grid {
                grid-template-columns: 1fr;
            }
            
            .info-item {
                flex-direction: column;
                align-items: flex-start;
                gap: 5px;
            }
            
            .info-value {
                align-self: stretch;
                text-align: center;
            }
        }
    </style>
</head>
<body>
    <?php include "header.php"; ?>
    
    <div class="container">
        <h1 class="page-title">系统概览</h1>
        
        <div class="main-content fade-in">
            <!-- 核心统计 -->
            <div class="dashboard-section">
                <h2 class="section-title">📊 核心统计</h2>
                <div class="stats-grid">
                    <div class="stat-card" onclick="location.href='duocang.php'">
                        <div class="stat-number"><?= $groupStats['total'] ?? 0 ?></div>
                        <div class="stat-label">多仓分组</div>
                        <div class="stat-subtext">
                            <?= $groupStats['with_local'] ?? 0 ?> 含本地资源 | 
                            <?= $groupStats['with_live'] ?? 0 ?> 含直播源
                        </div>
                    </div>
                    
                    <div class="stat-card info" onclick="location.href='repo.php'">
                        <div class="stat-number"><?= $repoStats['total'] ?? 0 ?></div>
                        <div class="stat-label">远程仓库</div>
                        <div class="stat-subtext">可用的数据源</div>
                    </div>
                    
                    <div class="stat-card purple" onclick="location.href='live.php'">
                        <div class="stat-number"><?= $liveStats['total'] ?? 0 ?></div>
                        <div class="stat-label">直播源</div>
                        <div class="stat-subtext">直播频道配置</div>
                    </div>
                    
                    <div class="stat-card orange" onclick="location.href='local.php'">
                        <div class="stat-number"><?= $localFileStats['total'] ?? 0 ?></div>
                        <div class="stat-label">本地文件</div>
                        <div class="stat-subtext">本地资源文件</div>
                    </div>
                    
                    <div class="stat-card teal" onclick="location.href='emby.php'">
                        <div class="stat-number"><?= $embyStats['total'] ?? 0 ?></div>
                        <div class="stat-label">Emby配置</div>
                        <div class="stat-subtext">媒体服务器配置</div>
                    </div>
                </div>
            </div>
            
            <!-- 快速操作 -->
            <div class="dashboard-section">
                <h2 class="section-title">🚀 快速操作</h2>
                <div class="quick-actions">
                    <a href="edit_repo_group.php" class="action-btn">
                        <span class="action-icon">📁</span>
                        <span>多仓分组管理</span>
                    </a>
                    <a href="edit_repos.php" class="action-btn">
                        <span class="action-icon">📦</span>
                        <span>远程仓库管理</span>
                    </a>
                    <a href="edit_lives.php" class="action-btn">
                        <span class="action-icon">📺</span>
                        <span>直播源管理</span>
                    </a>
                    <a href="local_repo.php" class="action-btn">
                        <span class="action-icon">💾</span>
                        <span>本地文件管理</span>
                    </a>
                    <a href="edit_emby.php" class="action-btn">
                        <span class="action-icon">🎬</span>
                        <span>Emby配置</span>
                    </a>
                    <a href="download.php" class="action-btn">
                        <span class="action-icon">📱</span>
                        <span>APK下载</span>
                    </a>
                </div>
            </div>
            
            <!-- 系统信息 -->
            <div class="dashboard-section">
                <h2 class="section-title">⚙️ 系统信息</h2>
                <div class="card">
                    <div class="system-info-grid">
                        <!-- 服务器信息组 -->
                        <div class="info-group">
                            <div class="info-group-title">
                                <span>🖥️</span>
                                <span>服务器信息</span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">PHP版本</span>
                                <span class="info-value"><?= $systemInfo['php_version'] ?></span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">服务器软件</span>
                                <span class="info-value"><?= $systemInfo['server_software'] ?></span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">时区设置</span>
                                <span class="info-value"><?= $systemInfo['timezone'] ?></span>
                            </div>
                        </div>
                        
                        <!-- 数据库信息组 -->
                        <div class="info-group">
                            <div class="info-group-title">
                                <span>🗄️</span>
                                <span>数据库信息</span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">MySQL版本</span>
                                <span class="info-value"><?= $systemInfo['mysql_version'] ?></span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">数据库名称</span>
                                <span class="info-value"><?= $config['db_name'] ?></span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">数据库主机</span>
                                <span class="info-value"><?= $config['db_host'] ?></span>
                            </div>
                        </div>
                        
                        <!-- 系统配置组 -->
                        <div class="info-group">
                            <div class="info-group-title">
                                <span>⚡</span>
                                <span>系统配置</span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">内存限制</span>
                                <span class="info-value"><?= $systemInfo['memory_limit'] ?></span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">上传限制</span>
                                <span class="info-value"><?= $systemInfo['upload_max_filesize'] ?></span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">执行超时</span>
                                <span class="info-value"><?= $systemInfo['max_execution_time'] ?> 秒</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- 时间显示 -->
                    <div class="time-display">
                        <div class="time-label">服务器当前时间</div>
                        <div class="time-value" id="currentTime"><?= date('Y-m-d H:i:s') ?></div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // 增强交互效果
        document.addEventListener('DOMContentLoaded', function() {
            // 统计卡片点击效果
            const statCards = document.querySelectorAll('.stat-card');
            statCards.forEach(card => {
                card.addEventListener('click', function() {
                    this.style.transform = 'scale(0.95)';
                    setTimeout(() => {
                        this.style.transform = '';
                    }, 150);
                });
            });
            
            // 实时更新时间
            function updateTime() {
                const now = new Date();
                document.getElementById('currentTime').textContent = 
                    now.toLocaleString('zh-CN', { 
                        year: 'numeric', 
                        month: '2-digit', 
                        day: '2-digit',
                        hour: '2-digit',
                        minute: '2-digit',
                        second: '2-digit'
                    });
            }
            
            // 每秒更新一次时间
            setInterval(updateTime, 1000);
        });
    </script>
</body>
</html>